// Waveguide Class

// QCD-audio FWF Research Project
// David Pirr, Katharina Vogt, Robert Hldrich
// (C) IEM 2009-2012

Waveguide {

	classvar <guides;

	var refNs, refWe, refUd;
	var vn, vs, ve, vw, vu, vd;
	var mesh;
	var lambdasqr, <>bound;
	var outArrayN, outArrayS, outArrayW, outArrayE, outArrayU, outArrayD;
	var size, ns;
	var coefArray; 
	var vnO, vsO, veO, vwO, vuO, vdO;
	var vnT, vsT, veT, vwT, vuT, vdT;
	var outBufN, <>outBufS, <>outBufW, <>outBufE, <>outBufU, <>outBufD;
	var state;
	var rMask;
	

	*initClass {
		guides = Array.new;
		UI.registerForShutdown({this.freeAll});	
	}

	*new {arg isize, icoef, ibound = -0.1, istate = 0;
		^super.new.init(isize,icoef,ibound,istate);
	}
	
	*freeAll {
		var list = guides.copy;
		guides = Array.new;
		list.do({arg item; item.free});	
	}

	init {arg isize, icoef, ibound, istate, iinC;
		size = isize;
		coefArray = icoef;
		ns = size - 1;
		state = istate;
				
		bound = ibound;
		
		mesh = Array.fillND([size-1,size-1,size-1], {arg r,c,p; 0.0});
		
		rMask = Array.fillND([size-1,size-1], {arg r,c; (1.0.rand-0.5)*2}); 
		
		vn = Array.fillND([size,size,size], {arg r,c,p,d; 0.0});
		vs = Array.fillND([size,size,size], {arg r,c,p,d; 0.0});
		ve = Array.fillND([size,size,size], {arg r,c,p,d; 0.0});
		vw = Array.fillND([size,size,size], {arg r,c,p,d; 0.0});
		vu = Array.fillND([size,size,size], {arg r,c,p,d; 0.0});
		vd = Array.fillND([size,size,size], {arg r,c,p,d; 0.0});
		
		vnO = vn.deepCopy;
		vsO = vs.deepCopy;
		veO = ve.deepCopy;
		vwO = vw.deepCopy;
		vuO = vu.deepCopy;
		vdO = vd.deepCopy;

		vnT = vn.deepCopy; 
		vsT = vs.deepCopy;
		veT = ve.deepCopy;
		vwT = vw.deepCopy;
		vuT = vu.deepCopy;
		vdT = vd.deepCopy;
		
		refNs = Array.fillND([ns-1,ns,ns], {arg r,c,p; 
								if (coefArray[r][c][p]!=coefArray[r+1][c][p]) {
									if (coefArray[r][c][p] == state) { 
										1.0;
									} {
										if (coefArray[r+1][c][p] == state) { 
											1.0.neg;
										} {
											0.0;	
										};
									}			
								} {
									0.0;
								};
							};
						);
							
		refWe = Array.fillND([ns,ns-1,ns], {arg r,c,p; 
								if (coefArray[r][c][p]!=coefArray[r][c+1][p]) {
									if (coefArray[r][c][p] == state) { 
										1.0;
									} {
										if (coefArray[r][c+1][p] == state) { 
											1.0.neg;
										} {
											0.0;	
										};
									}	
								} {
									0.0;
								};
								};
							);
							
		refUd = Array.fillND([ns,ns,ns-1], {arg r,c,p; 
								if (coefArray[r][c][p]!=coefArray[r][c][p+1]) {
									if (coefArray[r][c][p] == state) { 
										1.0;
									} {
										if (coefArray[r][c][p+1] == state) { 
											1.0.neg;
										} {
											0.0;	
										};
									}	
								} {
									0.0;
								};
								};
							);					
		
		outArrayN = [];		
		outArrayS = [];		
		outArrayW = [];		
		outArrayE = [];		
		outArrayU = [];		
		outArrayD = [];							
							
		lambdasqr = (1/3);
	}
	
	applyEx {arg inEx, plane = \N;
		var tempV, cond, vec;
		if (coefArray.shape.drop(1) == inEx.shape) {
			
			tempV = plane.switch(
				  \S, {cond = {arg r, c, p; (r == 0)}; vec = [1,2]; vn; },
				  \N, {cond = {arg r, c, p; (r == ns)}; vec = [1,2]; vs; },
				  \W, {cond = {arg r, c, p; (c == 0)}; vec = [0,2]; ve; },
				  \E, {cond = {arg r, c, p; (c == ns)}; vec = [0,2]; vw; },
				  \D, {cond = {arg r, c, p; (p == 0)}; vec = [0,1]; vu; },
				  \U, {cond = {arg r, c, p; (p == ns)}; vec = [0,1]; vd; }				  
			);
			
			size.do{arg r;
				size.do{arg c;
					size.do{arg p;
						if (cond.(r,c,p)  && (coefArray[r][c][p] == state)) {
							tempV[r][c][p] = inEx[vec[0]][vec[1]];
						};
					};
				};
			};
			
		} {
			"ATTENTION: Exitation matrix not in the right format!".postln;
		};
	}
	
	nextSample {
		var tempZ = 0, tempMo = 0, tempPo = 0;
		
		vnT = vn.deepCopy; 
		vsT = vs.deepCopy;
		veT = ve.deepCopy;
		vwT = vw.deepCopy;
		vuT = vu.deepCopy;
		vdT = vd.deepCopy;
		
		vnO = vn.deepCopy;
		vsO = vs.deepCopy;
		veO = ve.deepCopy;
		vwO = vw.deepCopy;
		vuO = vu.deepCopy;
		vdO = vd.deepCopy;	
		
		(ns-1).do{arg r;
			ns.do{arg c;
				ns.do{arg p;
					vnO[r+1][c][p] = ((1-refNs[r][c][p].squared)*vnT[r+1][c][p]) + 
								  (refNs[r][c][p]*vsT[r+1][c][p]);
					vsO[r+1][c][p] = ((1-refNs[r][c][p].squared)*vsT[r+1][c][p]) -
								  (refNs[r][c][p]*vnT[r+1][c][p]);				}
			}
		};
		
		ns.do{arg r;
			(ns-1).do{arg c;
				ns.do{arg p;
					veO[r][c+1][p] = ((1-refWe[r][c][p].squared)*veT[r][c+1][p]) + 
								  (refWe[r][c][p]*vwT[r][c+1][p]);
					vwO[r][c+1][p] = ((1-refWe[r][c][p].squared)*vwT[r][c+1][p]) - 
								  (refWe[r][c][p]*veT[r][c+1][p]);				}
			}
		};
		
		ns.do{arg r;
			ns.do{arg c;
				(ns-1).do{arg p;
					vuO[r][c][p+1] = ((1-refUd[r][c][p].squared)*vuT[r][c][p+1]) + 
								  (refUd[r][c][p]*vdT[r][c][p+1]);
					vdO[r][c][p+1] = ((1-refUd[r][c][p].squared)*vdT[r][c][p+1]) - 
								  (refUd[r][c][p]*vuT[r][c][p+1]);				}
			}
		};
		
		
		ns.do{arg r;
			ns.do{arg c;
				ns.do{arg p;
					mesh[r][c][p] = lambdasqr*(vnO[r][c][p] + 
										    veO[r][c][p] + 
										    vuO[r][c][p] + 
										    vsO[r+1][c][p] + 
										    vwO[r][c+1][p] + 
										    vdO[r][c][p+1]);
										    
					}
				}
			};
			
		ns.do{arg r;
			ns.do{arg c;
				ns.do{arg p;
					vn[r+1][c][p] = mesh[r][c][p] - vsO[r+1][c][p];
					ve[r][c+1][p] = mesh[r][c][p] - vwO[r][c+1][p];
					vu[r][c][p+1] = mesh[r][c][p] - vdO[r][c][p+1];
					
					vs[r][c][p] = mesh[r][c][p] - vnO[r][c][p];
					vw[r][c][p] = mesh[r][c][p] - veO[r][c][p];
					vd[r][c][p] = mesh[r][c][p] - vuO[r][c][p];
				}
			}
		};	
		
		size.do{arg i;
			size.do{arg j;
				vn[0][i][j] = bound*vsO[0][i][j];	
				vs[ns][i][j] = bound*vnO[ns][i][j];
				ve[i][0][j] = bound*vwO[i][0][j];	
				vw[i][ns][j] = bound*veO[i][ns][j];	
				vu[i][j][0] = bound*vdO[i][j][0];	
				vd[i][j][ns] = bound*veO[i][j][ns];	
			}
		};
		
			 		
		
		outArrayN = outArrayN.add((mesh.slice(ns-1,nil,nil)*rMask).sum.sum);
		outArrayS = outArrayS.add((mesh.slice(0,nil,nil)*rMask).sum.sum);
		outArrayE = outArrayE.add((mesh.slice(nil,ns-1,nil)*rMask).sum.sum);
		outArrayW = outArrayW.add((mesh.slice(nil,0,nil)*rMask).sum.sum);
		outArrayU = outArrayU.add((mesh.slice(nil,nil,ns-1)*rMask).sum.sum);
		outArrayD = outArrayD.add((mesh.slice(nil,nil,0)*rMask).sum.sum);
	}

	calc {arg server, samples;
		
		(samples).do{
			this.nextSample;
		};
		
		outArrayN = outArrayN.normalize(-1.0,1.0);
		outArrayS = outArrayS.normalize(-1.0,1.0);			outArrayE = outArrayE.normalize(-1.0,1.0);
		outArrayW = outArrayW.normalize(-1.0,1.0);
		outArrayU = outArrayU.normalize(-1.0,1.0);
		outArrayD = outArrayD.normalize(-1.0,1.0);			
		if (outBufN.isNil.not) {
			[outBufN,outBufS,outBufE,outBufW,outBufU,outBufD].do{arg item; item.free};
		};
		
		outBufN = Buffer.loadCollection(server,outArrayN,1);
		outBufS = Buffer.loadCollection(server,outArrayS,1);
		outBufE = Buffer.loadCollection(server,outArrayE,1);
		outBufW = Buffer.loadCollection(server,outArrayW,1);
		outBufU = Buffer.loadCollection(server,outArrayU,1);
		outBufD = Buffer.loadCollection(server,outArrayD,1);
										
		^[outBufN, outBufS, outBufE, outBufW, outBufU, outBufD];
	}

	play {arg sampleRate, dir;
		var tBuf = dir.switch(
				  \N, {outBufN},
				  \S, {outBufS},
				  \E, {outBufE},
				  \W, {outBufW},
				  \U, {outBufU},
				  \D, {outBufD}				  	
			);
		
		
		{
			Out.ar([0,1],PlayBuf.ar(1,tBuf.bufnum,
						BufRateScale.kr(tBuf.bufnum)*(sampleRate/44100),
						doneAction:2))
		}.play;
	}
	
	plot {
		var lenght = outArrayN.size;
		[outArrayN,outArrayS,outArrayE,outArrayW,outArrayU,outArrayD].lace(lenght)
			.plot(\N_S_E_W_U_D,numChannels:6);
	}
	
	
	free {
		this.outBufN.free;
		this.outBufS.free;
		this.outBufE.free;
		this.outBufW.free;
		this.outBufU.free;
		this.outBufD.free;						
	}


}



